#!/usr/bin/python3

"""Test cases for softwareproperties.AptAuth class."""

import os
import subprocess
import time
import unittest
import unittest.mock

import softwareproperties.AptAuth

KINETIC_DEFAULT = """\
tru:t:1:1658826897:0:3:1:5
pub:-:4096:1:D94AA3F0EFE21092:1336774248:::-:::scSC::::::23::0:
fpr:::::::::843938DF228D22F7B3742BC0D94AA3F0EFE21092:
uid:-::::1336774248::77355A0B96082B2694009775B6490C605BD16B6F::\
Ubuntu CD Image Automatic Signing Key (2012) <cdimage@ubuntu.com>::::::::::0:
pub:-:4096:1:871920D1991BC93C:1537196506:::-:::scSC::::::23::0:
fpr:::::::::F6ECB3762474EDA9D21B7022871920D1991BC93C:
uid:-::::1537196506::BE438F08F546424C0EA810FD722053597EB5127B::\
Ubuntu Archive Automatic Signing Key (2018) <ftpmaster@ubuntu.com>::::::::::0:
"""

KINETIC_SQ = """\
0. 843938DF228D22F7B3742BC0D94AA3F0EFE21092 Ubuntu CD Image Automatic Signing Key (2012) <cdimage@ubuntu.com>
1. F6ECB3762474EDA9D21B7022871920D1991BC93C Ubuntu Archive Automatic Signing Key (2018) <ftpmaster@ubuntu.com>\
"""

class TestAptAuth(unittest.TestCase):
    """Test softwareproperties.AptAuth class."""

    @staticmethod
    def setUp():
        os.environ["TZ"] = "UTC"
        time.tzset()

    @unittest.mock.patch("os.listdir")
    @unittest.mock.patch("subprocess.run")
    def test_list_sq(self, run_mock, listdir_mock):
        """Test softwareproperties.AptAuth.list."""
        apt_auth = softwareproperties.AptAuth.AptAuth()
        listdir_mock.return_value = ["demo-key.gpg", "second-key.gpg"]

        run_mock.return_value = subprocess.CompletedProcess(
            args=[],
            returncode=0,
            stdout=KINETIC_SQ,
        )

        self.assertEqual(
            apt_auth.list(),
            [
                "/etc/apt/trusted.gpg.d/demo-key.gpg\n"
                "  0. 843938DF228D22F7B3742BC0D94AA3F0EFE21092 Ubuntu CD Image Automatic Signing Key (2012) <cdimage@ubuntu.com>\n"
                "  1. F6ECB3762474EDA9D21B7022871920D1991BC93C Ubuntu Archive Automatic Signing Key (2018) <ftpmaster@ubuntu.com>",
                "/etc/apt/trusted.gpg.d/second-key.gpg\n"
                "  0. 843938DF228D22F7B3742BC0D94AA3F0EFE21092 Ubuntu CD Image Automatic Signing Key (2012) <cdimage@ubuntu.com>\n"
                "  1. F6ECB3762474EDA9D21B7022871920D1991BC93C Ubuntu Archive Automatic Signing Key (2018) <ftpmaster@ubuntu.com>"
            ],
        )


    @unittest.mock.patch("os.listdir")
    @unittest.mock.patch("subprocess.run")
    def test_list_gpg(self, run_mock, listdir_mock):
        """Test softwareproperties.AptAuth.list."""
        apt_auth = softwareproperties.AptAuth.AptAuth()
        listdir_mock.return_value = ["demo-key.gpg", "second-key.gpg"]

        gpgv_ret = subprocess.CompletedProcess(
            args=[],
            returncode=0,
            stdout=KINETIC_DEFAULT,
        )

        times_called = 0
        def raise_side_effect(self, *args, **more):
            nonlocal times_called
            times_called += 1
            if times_called % 2 == 0:
                return gpgv_ret
            else:
                raise OSError()

        run_mock.side_effect = raise_side_effect
        self.assertEqual(
            apt_auth.list(),
            [
                "/etc/apt/trusted.gpg.d/demo-key.gpg\n"
                "  843938DF228D22F7B3742BC0D94AA3F0EFE21092 Ubuntu CD Image Automatic Signing Key (2012) <cdimage@ubuntu.com>\n"
                "  F6ECB3762474EDA9D21B7022871920D1991BC93C Ubuntu Archive Automatic Signing Key (2018) <ftpmaster@ubuntu.com>",
                "/etc/apt/trusted.gpg.d/second-key.gpg\n"
                "  843938DF228D22F7B3742BC0D94AA3F0EFE21092 Ubuntu CD Image Automatic Signing Key (2012) <cdimage@ubuntu.com>\n"
                "  F6ECB3762474EDA9D21B7022871920D1991BC93C Ubuntu Archive Automatic Signing Key (2018) <ftpmaster@ubuntu.com>"
            ],
        )


if __name__ == "__main__":
    unittest.main()
