///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// compute:
//  \int_\Omega d_dr(u) r dr dz
//
// check value on the unit square [0,1]^2
//
//           d u_r 
//           -----  r dr dz
//            d r
//
// and u_r takes monomial values : 1, r, ...
//
#include "rheolef/rheolef.h"
using namespace rheolef;
using namespace std;

typedef Float (*function_type)(const point&);

struct list_type {
	string        name;
	function_type function;
	Float  expect;
};
Float monom_1    (const point& x) { return 1; }
Float monom_r   (const point& x) { return x[0]; }
Float monom_z   (const point& x) { return x[1]; }
Float monom_r2 (const point& x) { return x[0]*x[0]; }
Float monom_rz (const point& x) { return x[0]*x[1]; }
Float monom_z2 (const point& x) { return x[1]*x[1]; }

// axisymmetric rz

list_type rz_monom_list[] = {
	{"1",	monom_1, 	0},
	{"r",	monom_r,	0.5},
	{"z",	monom_z,	0},
	{"r2",	monom_r2,	Float(2)/3},
	{"rz",	monom_rz,	0.25},
	{"z2",	monom_z2,	0}
};
const unsigned int monom_max = sizeof(rz_monom_list)/sizeof(list_type);

void usage()
{
      cerr << "form_d_dx_tst: usage: form_d_dx_tst"
	   << " {-Igeodir}*"
	   << " -|mesh[.geo]"
	   << " [-monom string]"
	   << endl;
      cerr << "example:\n";
      cerr << "  form_d_dx_tst -I../data square -monom rz\n";
      exit (1);
}
int main(int argc, char**argv)
{
    //
    // load geometry and options
    //
    geo omega;  
    string approx1 = "P2";
    string approx2 = "P1";
    string monom_id = "";
    bool mesh_done = false;
    function_type monom_function = 0;
    unsigned int monom_idx = 0;
    string coord_sys = "rz";

    if (argc <= 1) usage() ;

    for (int i = 1; i < argc; i++ ) {

      if (argv [i][0] == '-' && argv [i][1] == 'I')
	  append_dir_to_rheo_path (argv[i]+2) ;
      else if (strcmp(argv[i], "-monom") == 0) {
	  monom_id = argv[++i];
          for (unsigned int i = 0; i < monom_max; i++) {
	      if (monom_id == rz_monom_list [i].name) {
		monom_function = rz_monom_list [i].function;
		monom_idx = i;
	      }
          }
      } else if (strcmp(argv[i], "-") == 0) {
	  // input geo on standard input
	  if (mesh_done) usage() ;
	  cerr << " ! load geo on stdin" << endl ;
	  cin >> omega ;
	  mesh_done = true ;
      } else {
	  // input geo on file
	  if (mesh_done) usage() ;
	  cerr << " ! load " << argv[i] << endl ;
	  omega = geo(argv[i], coord_sys);
	  mesh_done = true ;
      }
    }
    if (!mesh_done) usage() ;
    omega.set_coordinate_system (coord_sys);
    if (!monom_function) {
	error_macro("invalid monom identifier: " << monom_id);
    }
    cerr << "syscoord = " << omega.coordinate_system() << endl;
    cerr << "monom = " << monom_id << endl;
    space Vh(omega, approx1);
    space Qh(omega, approx2);
    field monom = interpolate (Vh, monom_function);
    field one(Qh,1);
    form  b (Vh,Qh,"d_dx0");
    Float result = b(monom, one);
    cerr << setprecision(numeric_limits<Float>::digits10)
         << "b(omega," << approx1 << "[" << monom_id << "]," << approx2 << "[1]) = " << double(result) << endl;
    Float expect = rz_monom_list[monom_idx].expect;
    Float tol = 1e-10;

    if (fabs(result - expect) <= tol) {
        cerr << "ok" << endl;
        return 0;
    } else {
        cerr << "but it was expected " << expect << endl;
	cerr << "and error = " << fabs(result - expect) << endl;
	cerr << "and tol = " << tol << endl;
        return 1;
    }
}
