% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/reprex-undo.R
\name{un-reprex}
\alias{un-reprex}
\alias{reprex_invert}
\alias{reprex_clean}
\alias{reprex_rescue}
\title{Un-render a reprex}
\usage{
reprex_invert(input = NULL, outfile = NULL, venue = c("gh", "so",
  "ds", "r"), comment = opt("#>"))

reprex_clean(input = NULL, outfile = NULL, comment = opt("#>"))

reprex_rescue(input = NULL, outfile = NULL,
  prompt = getOption("prompt"), continue = getOption("continue"))
}
\arguments{
\item{input}{Character. If has length one and lacks a terminating newline,
interpreted as the path to a file containing reprex code. Otherwise,
assumed to hold reprex code as character vector. If not provided, the
clipboard is consulted for input.}

\item{outfile}{Optional basename for output file. When \code{NULL}, no file is
left behind. If \code{outfile = "foo"}, expect an output file in current working
directory named \code{foo_clean.R}. If \code{outfile = NA}, expect on output file in
a location and with basename derived from \code{input}, if a path, or in
current working directory with basename derived from \code{\link[=tempfile]{tempfile()}}
otherwise.}

\item{venue}{Character. Must be one of the following (case insensitive):
\itemize{
\item "gh" for \href{https://github.github.com/gfm/}{GitHub-Flavored Markdown}, the
default
\item "so" for \href{https://stackoverflow.com/editing-help}{Stack Overflow Markdown}
\item "ds" for Discourse, e.g.,
\href{https://community.rstudio.com}{community.rstudio.com}. Note: this is
currently just an alias for "gh"!
\item "r" for a runnable R script, with commented output interleaved
\item "rtf" for \href{https://en.wikipedia.org/wiki/Rich_Text_Format}{Rich Text Format} (not supported for
un-reprexing)
\item "html" for an HTML fragment suitable for inclusion in a larger HTML
document (not supported for un-reprexing)
}}

\item{comment}{regular expression that matches commented output lines}

\item{prompt}{character, the prompt at the start of R commands}

\item{continue}{character, the prompt for continuation lines}
}
\value{
Character vector holding just the clean R code, invisibly
}
\description{
Recover clean, runnable code from a reprex captured in the wild and write it
to user's clipboard. The code is also returned invisibly and optionally
written to file. Three different functions address various forms of
wild-caught reprex.
}
\section{Functions}{
\itemize{
\item \code{reprex_invert}: Attempts to reverse the effect of \code{\link[=reprex]{reprex()}}. When
\code{venue = "r"}, this just becomes a wrapper around \code{reprex_clean()}.

\item \code{reprex_clean}: Assumes R code is top-level, possibly interleaved with
commented output, e.g., a displayed reprex copied from GitHub or the direct
output of \code{reprex(..., venue = "R")}. This function removes commented
output.

\item \code{reprex_rescue}: Assumes R code lines start with a prompt and that
printed output is top-level, e.g., what you'd get from copy/paste from the
R Console. Removes lines of output and strips prompts from lines holding R
commands.
}}

\examples{
\dontrun{
## a rendered reprex can be inverted, at least approximately
tmp_in <- file.path(tempdir(), "roundtrip-input")
x <- reprex({
  #' Some text
  #+ chunk-label-and-options-cannot-be-recovered, message = TRUE
  (x <- 1:4)
  #' More text
  y <- 2:5
  x + y
}, show = FALSE, advertise = FALSE, outfile = tmp_in)
tmp_out <- file.path(tempdir(), "roundtrip-output")
x <- reprex_invert(x, outfile = tmp_out)
x

# clean up
file.remove(
  list.files(dirname(tmp_in), pattern = "roundtrip", full.names = TRUE)
)
}
\dontrun{
## a displayed reprex can be cleaned of commented output
tmp <- file.path(tempdir(), "commented-code")
x <- c(
  "## a regular comment, which is retained",
  "(x <- 1:4)",
  "#> [1] 1 2 3 4",
  "median(x)",
  "#> [1] 2.5"
  )
out <- reprex_clean(x, outfile = tmp)
out

# clean up
file.remove(
  list.files(dirname(tmp), pattern = "commented-code", full.names = TRUE)
)

## round trip with reprex(..., venue = "R")
code_in <- c("x <- rnorm(2)", "min(x)")
res <- reprex(input = code_in, venue = "R", advertise = FALSE)
res
(code_out <- reprex_clean(res))
identical(code_in, code_out)
}
\dontrun{
## rescue a reprex that was copied from a live R session
tmp <- file.path(tempdir(), "live-transcript")
x <- c(
  "> ## a regular comment, which is retained",
  "> (x <- 1:4)",
  "[1] 1 2 3 4",
  "> median(x)",
  "[1] 2.5"
)
out <- reprex_rescue(x, outfile = tmp)
out

# clean up
file.remove(
  list.files(dirname(tmp),pattern = "live-transcript", full.names = TRUE)
)
}
}
